<?php

namespace App\Livewire\Settings;

use App\Models\MaintenanceType;
use Livewire\Component;
use Livewire\WithPagination;

class MaintenanceTypeIndex extends Component
{
    use WithPagination;

    public bool $isEditing = false;

    public ?int $maintenanceTypeId = null;

    public string $name = '';

    public ?string $description = null;

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255|unique:maintenance_types,name,'.$this->maintenanceTypeId,
            'description' => 'nullable|string|max:500',
        ];
    }

    public function create()
    {
        $this->isEditing = false;
        $this->reset(['name', 'description', 'maintenanceTypeId']);
        $this->dispatch('open-modal');
    }

    public function edit(MaintenanceType $maintenanceType)
    {
        $this->isEditing = true;
        $this->maintenanceTypeId = $maintenanceType->id;
        $this->name = $maintenanceType->name;
        $this->description = $maintenanceType->description;
        $this->dispatch('open-modal');
    }

    public function save()
    {
        $this->validate();

        $data = [
            'name' => $this->name,
            'description' => $this->description,
        ];

        if ($this->isEditing) {
            $maintenanceType = MaintenanceType::findOrFail($this->maintenanceTypeId);
            $maintenanceType->update($data);
            session()->flash('flash.banner', 'Maintenance type updated successfully!');
        } else {
            MaintenanceType::create($data);
            session()->flash('flash.banner', 'Maintenance type created successfully!');
        }

        $this->dispatch('close-modal');
    }

    public function delete(MaintenanceType $maintenanceType)
    {
        // Optional: Check if the type is in use before deleting
        // if ($maintenanceType->maintenanceRecords()->count() > 0) {
        //     session()->flash('flash.banner', 'This maintenance type is in use and cannot be deleted.');
        //     session()->flash('flash.bannerStyle', 'danger');
        //     return;
        // }

        $maintenanceType->delete();
        session()->flash('flash.banner', 'Maintenance type deleted successfully.');
    }

    public function render()
    {
        $maintenanceTypes = MaintenanceType::latest()->paginate(10);

        return view('livewire.settings.maintenance-type-index', [
            'maintenanceTypes' => $maintenanceTypes,
        ]);
    }
}
